/*
** Copyright (C) 2001,2002 Sacha Faust <sacha@severus.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
package org.ideahamster.metis.report;

import org.ideahamster.metis.util.*;
import org.ideahamster.metis.stat.*;
import faust.sacha.web.data.*;
import faust.sacha.web.util.*;
import java.io.*;
import java.util.*;

/**
 * @author Sacha Faust
 * @version <PRE>1.3</PRE>
 */
public class MetisReportHTML extends MetisReport{

    private File m_reportFolder = null;
    private String BLANK_LINE = "&nbsp;";
    private String DATA_FOLDERNAME = "Data";
    
    
    public MetisReportHTML( Site site, SiteStats siteStats ){
        super(site, siteStats);
        m_site = site;
        m_siteStats = siteStats;

    }
//------------------------------------------------------------------------------
    private boolean createReportFolder( String name ){
        int id = 1;
        m_reportFolder = new File(name);
        
        while( m_reportFolder.exists() ){
         
            m_reportFolder.renameTo( new File(m_reportFolder.getName() + "_" + id) );
            
            if( id == Integer.MAX_VALUE ){
                System.err.println("To many folders for this domain present!");
                return false;
            }
            else
                id++;
        }
        
        return m_reportFolder.mkdir();
    }
//------------------------------------------------------------------------------
    private String generateFramePage(){
        StringWriter reportBuff = new StringWriter();
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<head>" + "\n" );
        reportBuff.write( "<title>Metis report for " + m_site.getDomain() + "</title>" + "\n" );
        reportBuff.write( "</head>" + "\n" );
        
        reportBuff.write( "<frameset rows=\"56,602*\" cols=\"*\" frameborder=\"YES\" border=\"0\" framespacing=\"0\" bordercolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<frame name=\"topFrame\" scrolling=\"NO\" noresize src=\"top_frame.htm\" >" + "\n" );
        reportBuff.write( "<frameset cols=\"80,*\" frameborder=\"YES\" bordercolor=\"#CCCCCC\"> " + "\n" );
        reportBuff.write( "<frame name=\"leftFrame\" noresize scrolling=\"NO\" src=\"left_frame.htm\">" + "\n" );
        reportBuff.write( "<frame name=\"mainFrame\" src=\"center_frame.htm\">" + "\n" );
        reportBuff.write( "</frameset>" + "\n" );
        reportBuff.write( "</frameset>" + "\n" );
        reportBuff.write( "<noframes>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\" bordercolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "</body></noframes>" + "\n" );
        reportBuff.write( "</html>" );
        
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------
    private String generateLeftFrame(){
        StringWriter reportBuff = new StringWriter();
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<div align=\"center\">" + "\n" );
        reportBuff.write( "<p><a href=\"center_frame.htm\" target=\"mainFrame\">General</a></p>" + "\n" );
        reportBuff.write( "<p><a href=\"" + DATA_FOLDERNAME + File.separator + "file_by_type.htm\" target=\"mainFrame\">Files by Type</a> </p>" + "\n" );
        reportBuff.write( "<p><a href=\"" + DATA_FOLDERNAME + File.separator + "folder.htm\" target=\"mainFrame\">Folders</a> </p>" + "\n" );
        reportBuff.write( "<p><a href=\"" + DATA_FOLDERNAME + File.separator + "file_info_by_folder.htm\" target=\"mainFrame\">File Info by Folders</a> </p>" + "\n" );
        reportBuff.write( "<p><a href=\"" + DATA_FOLDERNAME + File.separator + "comments_by_file.htm\" target=\"mainFrame\">Comments by Files</a> </p>" + "\n" );        
        reportBuff.write( "<p><a href=\"" + DATA_FOLDERNAME + File.separator + "emails_by_domain.htm\" target=\"mainFrame\">Emails by Domain</a> </p>" + "\n" );
        reportBuff.write( "<p><a href=\"" + DATA_FOLDERNAME + File.separator + "site_scripts.htm\" target=\"mainFrame\">Site scripts</a> </p>" + "\n" );
        reportBuff.write( "<p><a href=\"" + DATA_FOLDERNAME + File.separator + "outside_url_by_domain.htm\" target=\"mainFrame\">Outside urls by domain</a> </p>" + "\n" );
        reportBuff.write( "<p><a href=\"" + DATA_FOLDERNAME + File.separator + "site_map.htm\" target=\"mainFrame\">Site Map</a> </p>" + "\n" );
        reportBuff.write( "</div>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------
    private String generateTopFrame(){
        StringWriter reportBuff = new StringWriter();
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<div align=\"center\"><font size=\"+2\">" + m_site.getDomain() + " Metis report.</font> </div>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------
    private String getHTMLLinkCode( String link ){
        return "<a href=\"" + link + "\">" + link + "</a>";
    }
//------------------------------------------------------------------------------
    private String getHTMLLinkCode( String link, String text ){
        return "<a href=\"" + link + "\">" + text + "</a>";
    }
//------------------------------------------------------------------------------
    private String getHTMLEmailLinkCode( String link ){
        return "<a href=\"mailto:" + link + "\">" + link + "</a>";
    }
//------------------------------------------------------------------------------
    private String getHTMLEmailLinkCode( EmailURL email ){
        return getHTMLEmailLinkCode(email.getEmailURL());
    }
//------------------------------------------------------------------------------    
    protected String generateHeaderReport() {
        StringWriter reportBuff = new StringWriter();
        double totalSiteSize = 0;
        String totalSiteSizeStr = null;
        int dotPos = 0;
      
        /*
         *  Note:
         *      this <html> and the following <body> tags will be ended
         *      in generateFootNote()
         */
        reportBuff.write("<html>" + "\n"); // 
        reportBuff.write("<body bgcolor=\"#FFFFFF\">" + "\n");
        reportBuff.write("<p align=\"left\">Site : " + m_site.getDomain() + "</p>" + "\n");
        reportBuff.write("<p align=\"left\">Server type : " + m_siteStats.getSiteServerType() + "</p>" + "\n");
        reportBuff.write("<p align=\"left\">Date : " + new Date().toString() + "\n" );
        reportBuff.write("<p align=\"left\">Total Number of Files : " + m_siteStats.getNumberOfFiles() + "\n" );
        reportBuff.write("<p align=\"left\">Total Number of Folders : " + m_siteStats.getNumberOfFolders() + "\n" );
        reportBuff.write("<p align=\"left\">Total Number of E-mails : " + m_siteStats.getNumberOfEmails() + "\n" );
        try{
            totalSiteSize = (double)m_siteStats.getTotalObjectSize();
            totalSiteSizeStr = String.valueOf( totalSiteSize / (double)1048576 );
            dotPos = totalSiteSizeStr.indexOf(".");
            if( dotPos != -1 ){
                if( (totalSiteSizeStr.length() - dotPos) > 2 )
                    totalSiteSizeStr = totalSiteSizeStr.substring(0, dotPos + 2);
            }
            reportBuff.write("<p align=\"left\">Site size estimate : " + totalSiteSizeStr + " MB" + "\n" );
        }
        catch( NumberFormatException numEx ){
            if( WebGlobal.DEBUG )
                System.err.println( "MetisReportHTML::generateHeaderReport() : " + numEx );
            
            
            reportBuff.write("<p align=\"left\">Site size estimate : " + m_siteStats.getTotalObjectSize() + " Bytes" + "\n" );
        }
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------
     protected String generateFootNote(){
         StringWriter reportBuff = new StringWriter();
         
         // spacing
         reportBuff.write( "<p>" + BLANK_LINE + "</p>" + "\n");
         reportBuff.write( "<p>" + BLANK_LINE + "</p>" + "\n");
         reportBuff.write( "<p>" + BLANK_LINE + "</p>" + "\n");

         reportBuff.write( "<hr align=\"center\" noshade>" + "\n" );
         reportBuff.write( "<p><font size=\"2\">This report was generated by " +
                            MetisGlobal.METIS_USER_AGENT + " written by " + MetisGlobal.METIS_AUTHOR + " : </font></p>"  );
         reportBuff.write( getHTMLEmailLinkCode(MetisGlobal.METIS_AUTHOR_EMAIL) + "\n" );
         
         reportBuff.write( "</body>" + "\n" );
         reportBuff.write( "</html>" + "\n" );
         
         return reportBuff.toString();
     }    
//------------------------------------------------------------------------------    
    protected String generateEmailByDomainReport() {
        StringWriter reportBuff = new StringWriter();
        EmailByDomainStat emailStat = m_siteStats.getEmailByDomainStats();
        Enumeration domainsKeys = emailStat.getDataKeys();
        String domain = null;
        ElementStat elementData = null;
        ArrayList elementArray = null;
        EmailURL email = null;        
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<table width=\"75%\" border=\"1\" bordercolor=\"#FFFFFF\">" + "\n" );
        
        while( domainsKeys.hasMoreElements() ){
            domain = (String)domainsKeys.nextElement();
            elementData = emailStat.getElement(domain);
            elementArray = elementData.getMembers();
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Domain</b></td>" + "\n" );
            reportBuff.write( "<td>" + domain + "</td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n");
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n");
            reportBuff.write( "</tr>" + "\n" );
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>E-mails</b></td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "<td><b>Subject</b></td>" + "\n" );
            reportBuff.write( "<td><b>Arguments</b></td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            for( int i = 0; i < elementArray.size(); i++ ){
                email = (EmailURL)elementArray.get(i);
                
                reportBuff.write( "<tr>" + "\n" );
                reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                reportBuff.write( "<td>" + getHTMLEmailLinkCode(email) + "</td>" + "\n" );
                reportBuff.write( "<td>" + email.getSubject() + "</td>" + "\n" );
                reportBuff.write( "<td>" + email.getArgumentData() + "</td>" + "\n" );
                reportBuff.write( "</tr>" + "\n" );
            }
        }
        reportBuff.write( "</table>"  + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------    
    protected String generateFileInfoReportByFolder() {
        StringWriter reportBuff = new StringWriter();
        FolderStat folderStat = m_siteStats.getFolderStats();
        Enumeration folderKeys = folderStat.getDataKeys();
        String key = null;
        ElementStat elementData = null;
        ArrayList fileList = null;
        FileURL file = null;
        ArrayList queryList = null;
        ScriptStateInfo queryData = null;        
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<table width=\"75%\" border=\"1\" bordercolor=\"#FFFFFF\">" + "\n" );
        
        while( folderKeys.hasMoreElements() ){
            key = (String)folderKeys.nextElement();
            elementData = folderStat.getElement(key);
            fileList = elementData.getMembers();
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Folder Name</b></td>" + "\n" );
            reportBuff.write( "<td>" + key + "</td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Files</b></td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "<td><b>File Name</b></td>" + "\n" );
            reportBuff.write( "<td><b>Size</b></td>" + "\n" );
            reportBuff.write( "<td><b>Last Modified</b></td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            for( int i = 0; i < fileList.size(); i++ ){
                file = (FileURL)fileList.get(i);
                reportBuff.write( "<tr>" + "\n" );
                reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                reportBuff.write( "<td>" + getHTMLLinkCode(file.toString(), file.getFileName()) + "</td>" + "\n" );
                if( file.isScript() ){
                    /*
                     *  finishing the table entry
                     */
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                    reportBuff.write( "</tr>" + "\n" );
                    
                    reportBuff.write( "<tr>" + "\n" );
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                    reportBuff.write( "<td><b>Queries</b></td>"  + "\n" );
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                    queryList = file.getQueryDataList();
                    
                    for( int queryIndex = 0; queryIndex < queryList.size(); queryIndex++ ){
                        queryData = (ScriptStateInfo)queryList.get(queryIndex);
                        
                        reportBuff.write( "<tr>" + "\n" );
                        reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                        reportBuff.write( "<td>" + getHTMLLinkCode( queryData.getURL(), queryData.getQuery() ) + "</td>" + "\n" );
                        reportBuff.write( "<td>" + queryData.getContentLengthInfoAsString() + "</td>" + "\n" );
                        reportBuff.write( "<td>" + queryData.getLastModifiedInfo() + "<td>" + "\n" );
                        reportBuff.write( "</tr>" + "\n" );
                    }   
                }
                else{
                    reportBuff.write( "<td>" + file.getContentLengthInfoAsString() + "</td>" + "\n" );
                    reportBuff.write( "<td>" + file.getLastModifiedInfo() + "<td>" + "\n" );
                    reportBuff.write( "</tr>" + "\n" );
                }
            }
        }
        reportBuff.write( "</table>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------    
    protected String generateFolderReport() {
        StringWriter reportBuff = new StringWriter();
        FolderStat folderStat = m_siteStats.getFolderStats();
        Enumeration folderKeys = folderStat.getDataKeys();
        String key = null;
        ElementStat elementData = null;        
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        
        reportBuff.write( "<table width=\"75%\" border=\"1\" bordercolor=\"#FFFFFF\">" + "\n" );
        
        while( folderKeys.hasMoreElements() ){
            key = (String)folderKeys.nextElement();
            elementData = folderStat.getElement(key);
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Folder name</b></td>" + "\n" );
            reportBuff.write( "<td>" + getHTMLLinkCode(m_site.getInfo().getBaseURL() + key, key) + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Number of files</b></td>" + "\n" );
            reportBuff.write( "<td>" + elementData.getNumberOfElements() + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>% on Total</b></td>" + "\n" );
            reportBuff.write( "<td>" + Double.toString(elementData.getPercentage()) + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            //spacing
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
        }
        reportBuff.write( "</table>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------    
    protected String generateSiteMapReport() {
        StringWriter reportBuff = new StringWriter();
        Enumeration folderKeys = m_site.getFoldersNames();
        String folderName = null;
        ArrayList fileArray = null;
        FileURL file = null;
        ArrayList queryList = null;
        ScriptStateInfo queryData = null;
        
        reportBuff.write( m_site.getSiteURL() + "\n" );
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );        
     
        reportBuff.write( getHTMLLinkCode(m_site.getInfo().getBaseURL()) + "\n" );
        reportBuff.write( "<blockquote>" + "\n" );
        
        while( folderKeys.hasMoreElements() ){
            folderName = (String)folderKeys.nextElement();
            fileArray = m_site.getFolderElements(folderName);
        
            //folder
            reportBuff.write( getHTMLLinkCode(m_site.getInfo().getBaseURL() + folderName) + "\n" );
            reportBuff.write( "<blockquote>" + "\n" );
            
            //files
            for( int i = 0; i < fileArray.size(); i++ ){
                file = (FileURL)fileArray.get(i);
                //queryList = file.getQ
                queryList = file.getQueryDataList();
                
                reportBuff.write( (i == 0 ? "" : "<br>") + getHTMLLinkCode(file.toString()) + "\n" );
                //reportBuff.write( "<br>" + getHTMLLinkCode(file.toString()) + "\n" );
                
                for( int queryIndex = 0; queryIndex < queryList.size(); queryIndex++ ){
                
                    queryData = (ScriptStateInfo)queryList.get(queryIndex);
                    
                    if( queryIndex == 0 )
                        reportBuff.write( "<blockquote>" + "\n" );
                    
                    reportBuff.write( (queryIndex == 0 ? "" : "<br>") + getHTMLLinkCode(queryData.getURL()) + "\n" );
                }
                
                if( file.isScript() ){
                    //query
                    reportBuff.write( "</blockquote>" + "\n" );
                }
            }
            //folder
            reportBuff.write( "</blockquote>" + "\n" );
        }
        
        reportBuff.write( "</blockquote>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString(); 
    }
//------------------------------------------------------------------------------    
    protected String generateFileByTypeReport() {
        StringWriter reportBuff = new StringWriter();
        FileExtensionStat fileExStats = m_siteStats.getFileExtensionStats();
        Enumeration keys = fileExStats.getDataKeys();
        String fileExKey = null;
        ElementStat elementData = null;
        ArrayList fileList = null;
        FileURL file = null;
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<table width=\"75%\" border=\"1\" bordercolor=\"#FFFFFF\">" );
        
        while( keys.hasMoreElements() ){
            fileExKey = (String)keys.nextElement();
            elementData = (ElementStat)fileExStats.getElement( fileExKey );
            fileList = elementData.getMembers();
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Name</b></td>" + "\n" );
            reportBuff.write( "<td>" + fileExKey + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Number of files</b></td>" + "\n" );
            reportBuff.write( "<td>" + elementData.getNumberOfElements() + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>% on Total</b></td>" + "\n" );
            reportBuff.write( "<td>" + Double.toString(elementData.getPercentage()) + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>File List</b></td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            for( int i = 0; i < fileList.size(); i++ ){
                file = (FileURL)fileList.get(i);
                
                reportBuff.write( "<tr>" + "\n" );
                reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                reportBuff.write( "<td>" + getHTMLLinkCode(file.toString(), file.getPath() ) + "</td>" + "\n" );
                reportBuff.write( "</tr>" + "\n" );
            }
        }
        reportBuff.write( "</table>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------
     protected String generateCommentsByFileReport() {
        StringWriter reportBuff = new StringWriter();
        Enumeration folderKeys = m_site.getFoldersNames();
        String key = null;
        ArrayList fileList = null;
        FileURL file = null;
        ArrayList commentList = null;
        ArrayList queryList = null;
        String query = null;
        ScriptStateInfo queryData = null;
        String comment = null;
        
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<table width=\"75%\" border=\"1\" bordercolor=\"#FFFFFF\">" + "\n" );
        
        while( folderKeys.hasMoreElements() ){
            key = (String)folderKeys.nextElement();
            fileList = m_site.getFolderElements(key);
            for( int fileIndex = 0; fileIndex < fileList.size(); fileIndex++ ){
                file = (FileURL)fileList.get(fileIndex);
                
                if( !file.isScript() ){
                    commentList = file.getComments();
                    if( commentList.size() == 0 )
                        continue;
                }
                
                reportBuff.write( "<tr>" + "\n" );
                reportBuff.write( "<td><b>File name</b></td>" + "\n" );
                reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                reportBuff.write( "</tr>" + "\n" );
                
                reportBuff.write( "<tr>" + "\n" );
                reportBuff.write( "<td>" + getHTMLLinkCode(file.toString(), file.getFolder() + file.getFileName()) + "</td>" + "\n" );
                
                if( file.isScript() )
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                else
                    reportBuff.write( "<td><b>Comments</b></td>" + "\n" );

                reportBuff.write( "</tr>" + "\n" );
                
                if( file.isScript() ){
                    queryList = file.getQueryDataList();
                    
                    for(int queryIndex = 0; queryIndex < queryList.size(); queryIndex++ ){
                        queryData = (ScriptStateInfo)queryList.get(queryIndex);
                        
                        
                        if( queryData == null ){
                            System.err.println("MetisReportHTML::generateCommentByFileReport() : queryData is null");
                            continue;
                        }
                        
                        if( (query = queryData.getQuery()) == "" ){
                            System.err.println("MetisReportHTML::generateCommentByFileReport() : query is empty on : " + queryData.getURL() );
                            continue;
                        }
                        
                        reportBuff.write( "<tr>" + "\n" );
                        reportBuff.write( "<td><b>Query</b></td>" + "\n" );
                        reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                        reportBuff.write( "</tr>" + "\n" );
                        
                        reportBuff.write( "<tr>" + "\n" );
                        reportBuff.write( "<td>" + getHTMLLinkCode(queryData.getURL(), file.getFolder() + file.getFileName() + query) + "</td>" + "\n" );
                        reportBuff.write( "<td><b>Comments</b></td>" + "\n" );
                        reportBuff.write( "</tr>" + "\n" );
                        
                        commentList = queryData.getComments();
                        for( int commentIndex = 0; commentIndex < commentList.size(); commentIndex++ ){
                            comment = (String)commentList.get(commentIndex);
                            
                            reportBuff.write( "<tr>" + "\n" );
                            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                            // adding <xmp></xmp> to make sure we don't include html that is in the comment
                            // Tks Prizm for pointing out this tag and make me dump <pre>
                            reportBuff.write( "<td><xmp>" + comment + "</xmp></td>" + "\n" );
                            reportBuff.write( "</tr>" + "\n" );                            
                        }
                    }
                }
                for( int commentIndex = 0; commentIndex < commentList.size(); commentIndex++ ){
                    comment = (String)commentList.get(commentIndex);
                           
                    reportBuff.write( "<tr>" + "\n" );
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                    // adding <xmp></xmp> to make sure we don't include html that is in the comment
                    // Tks Prizm for pointing out this tag and make me dump <pre>
                    reportBuff.write( "<td><xmp>" + comment + "</xmp></td>" + "\n" );
                    reportBuff.write( "</tr>" + "\n" );                            
                }
            }
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
        }
        
        reportBuff.write( "</table>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
     }
//------------------------------------------------------------------------------
    protected String generateOutsideURLByDomainReport(){
        StringWriter reportBuff = new StringWriter();
        OutsideURLByDomainStat urlStat = m_siteStats.getOutsideURLByDomainStats();
        Enumeration domainKeys = urlStat.getDataKeys();
        String key = null;
        ElementStat elementData = null;
        ArrayList urlList = null;
        URLInfo url = null;
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<table width=\"%75\" border=\"1\" bordercolor=\"#FFFFFF\">" + "\n" );
        
        while( domainKeys.hasMoreElements() ){
            key = (String)domainKeys.nextElement();
            elementData = urlStat.getElement(key);
            urlList = elementData.getMembers();
            
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Domain</b></td>" + "\n" );
            reportBuff.write( "<td>" + key + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            for( int i = 0; i < urlList.size(); i++ ){
                url = (URLInfo)urlList.get(i);
                reportBuff.write( "<tr>" + "\n" );
                reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                reportBuff.write( "<td>" + getHTMLLinkCode(url.getURL()) + "<td>" + "\n" );
                reportBuff.write( "</tr>" + "\n" );
            }
        }
        reportBuff.write( "</table>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
    }
//------------------------------------------------------------------------------    
    protected String generateScriptReport(){       
        StringWriter reportBuff = new StringWriter();
        SiteScripts siteScripts = m_siteStats.getSiteScripts();
        ArrayList scriptList = siteScripts.getScripts();
        FileURL script = null;
        Enumeration varNames = null;
        String varKey = null;
        ArrayList dataList = null;
        String data = null;
        
        reportBuff.write( "<html>" + "\n" );
        reportBuff.write( "<body bgcolor=\"#FFFFFF\">" + "\n" );
        reportBuff.write( "<table width=\"%75\" border=\"1\" bordercolor=\"#FFFFFF\">" + "\n" );
        
        for( int scriptIndex = 0; scriptIndex < scriptList.size(); scriptIndex++ ){
            script = (FileURL)scriptList.get(scriptIndex);
            varNames = script.getArgumentVarNames();
            
            //show the script in folder/script format
            reportBuff.write( "<tr>" + "\n" );
            reportBuff.write( "<td><b>Script file</b></td>" + "\n" );
            reportBuff.write( "<td>" + getHTMLLinkCode(script.toString(), script.getFolder() + script.getFileName()) + "</td>" + "\n" );
            reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
            reportBuff.write( "</tr>" + "\n" );
            
            while( varNames.hasMoreElements() ){
                varKey = (String) varNames.nextElement();
                dataList = script.getArgumentVarDataList(varKey);

                reportBuff.write( "<tr>" + "\n" );
                reportBuff.write( "<td><b>Variable name</b></td>" + "\n" );
                reportBuff.write( "<td>" + varKey + "</td>" + "\n" );
                reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                reportBuff.write( "</tr>" + "\n" );
            
                reportBuff.write( "<tr>" + "\n" );
                reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                reportBuff.write( "<td><b>Instances</b></td>" + "\n" );
                reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                reportBuff.write( "</tr>" + "\n" );
                    
                for( int dataIndex = 0; dataIndex < dataList.size(); dataIndex++ ){
                    data = (String)dataList.get(dataIndex);
                    if( data == "" )
                        continue;
                    
                    reportBuff.write( "<tr>" + "\n" );
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                    reportBuff.write( "<td>" + BLANK_LINE + "</td>" + "\n" );
                    reportBuff.write( "<td>" + data + "</td>" + "\n" );
                    reportBuff.write( "</tr>" + "\n" );
                }
            }
        }
        reportBuff.write( "</table>" + "\n" );
        reportBuff.write( "</body>" + "\n" );
        reportBuff.write( "</html>" + "\n" );
        
        return reportBuff.toString();
    }


//------------------------------------------------------------------------------
     /*
      * Overload parent to handle proper files creation
      */
     public void generateReport(){
         File dataFolder = null;
         RandomAccessFile topFrame = null;
         RandomAccessFile leftFrame = null;
         RandomAccessFile centerFrame = null;
         RandomAccessFile emailsByDomainReport = null;
         RandomAccessFile fileByTypeReport = null;
         RandomAccessFile fileInfoByFolderReport = null;
         RandomAccessFile commentInfoByFileReport = null;
         RandomAccessFile folderReport = null;
         RandomAccessFile outsideUrlByDomainReport = null;
         RandomAccessFile siteMapReport = null;
         RandomAccessFile siteScriptsReport = null;
         
         if( !createReportFolder(m_site.getDomain()) ){
            System.err.println("Unable to create report folder!");
            System.exit(1);
         }
         
         dataFolder = new File(m_reportFolder, DATA_FOLDERNAME);
         if( !dataFolder.mkdir() ){
            System.err.println("Unable to create report data folder!");
            System.exit(1);             
         }
         
         try{
             super.setFile( new RandomAccessFile( new File(m_reportFolder, m_site.getDomain() + ".htm"), "rw") );
             topFrame = new RandomAccessFile( new File(m_reportFolder, "top_frame.htm"), "rw" );
             leftFrame = new RandomAccessFile( new File(m_reportFolder, "left_frame.htm"), "rw" );
             centerFrame = new RandomAccessFile( new File(m_reportFolder, "center_frame.htm"), "rw" );
             
             emailsByDomainReport = new RandomAccessFile( new File(dataFolder, "emails_by_domain.htm"), "rw" );
             fileByTypeReport = new RandomAccessFile( new File(dataFolder, "file_by_type.htm"), "rw" );
             fileInfoByFolderReport = new RandomAccessFile( new File(dataFolder, "file_info_by_folder.htm"), "rw" );
             commentInfoByFileReport = new RandomAccessFile( new File(dataFolder, "comments_by_file.htm"), "rw" ); 
             folderReport = new RandomAccessFile( new File(dataFolder, "folder.htm"), "rw" );
             outsideUrlByDomainReport = new RandomAccessFile( new File(dataFolder, "outside_url_by_domain.htm"), "rw" );
             siteMapReport = new RandomAccessFile( new File(dataFolder, "site_map.htm"), "rw" );
             siteScriptsReport = new RandomAccessFile( new File(dataFolder, "site_scripts.htm"), "rw" );
         }
         catch( FileNotFoundException notFoundEx ){
            System.err.println(notFoundEx);
            System.exit(1);
         }
         
         try{
             m_reportFile.writeChars( generateFramePage() );
             topFrame.writeChars( generateTopFrame() );
             leftFrame.writeChars( generateLeftFrame() );
             centerFrame.writeChars( generateHeaderReport() );
             centerFrame.writeChars( generateFootNote() );
             
             emailsByDomainReport.writeChars( generateEmailByDomainReport() );
             fileByTypeReport.writeChars( generateFileByTypeReport() );
             fileInfoByFolderReport.writeChars( generateFileInfoReportByFolder() );
             commentInfoByFileReport.writeChars( generateCommentsByFileReport() );
             folderReport.writeChars( generateFolderReport() );
             outsideUrlByDomainReport.writeChars( generateOutsideURLByDomainReport() );
             siteMapReport.writeChars( generateSiteMapReport() );
             siteScriptsReport.writeChars( generateScriptReport() );
         }
         catch( IOException ioExWrite ){
             System.err.println(ioExWrite);
         }
         finally{
             try{
                 m_reportFile.close();
                 topFrame.close();
                 leftFrame.close();
                 centerFrame.close();
                 centerFrame.close();

                 emailsByDomainReport.close();
                 fileByTypeReport.close();
                 fileInfoByFolderReport.close();
                 commentInfoByFileReport.close();
                 folderReport.close();
                 outsideUrlByDomainReport.close();
                 siteMapReport.close();
                 siteScriptsReport.close();
             }
             catch( IOException ioExClose ){
             }
         }
     }
//------------------------------------------------------------------------------     
}
