/*
** Copyright (C) 2001,2002 Sacha Faust <sacha@severus.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
package org.ideahamster.metis;

import faust.sacha.web.bot.spider.*;
import faust.sacha.web.bot.spider.event.*;
import faust.sacha.web.bot.spider.util.EnginGlobal;
import faust.sacha.web.data.*;
import faust.sacha.web.util.*;

import org.ideahamster.metis.stat.*;
import org.ideahamster.metis.report.*;
import org.ideahamster.metis.util.*;

import gnu.getopt.Getopt;
import java.net.MalformedURLException;

import java.util.Enumeration;

/** This is the main file for Metis.
 * @author Sacha Faust
 * @version <PRE>1.3</PRE>
 */
public class Metis {

    private Site m_site;
    private SpiderManager m_spiderManager;
    private SiteStats m_siteStats;
    private KeyboardProcessor m_keyboardProcessor;
    
//------------------------------------------------------------------------------
    public Metis() throws MalformedURLException {
        m_spiderManager = new SpiderManager( MetisOpt.getrootURL() );
        m_site = new Site( MetisOpt.getrootURL(), m_spiderManager );        
        m_keyboardProcessor = new KeyboardProcessor( System.in, m_spiderManager );
        
        SpiderManager.PRINT_SITE_START_MESSAGE = true;
        SpiderManager.PRINT_SITE_FINISH_MESSAGE = true;
        
    }
//------------------------------------------------------------------------------    
    public void run(){
        Site site = null;
        Enumeration keys = null;
        String siteName = null;
        SiteStats siteStats = null;
        m_keyboardProcessor.start();
        
        collectData();
        
        keys = m_spiderManager.getSiteNames();
        
        /*
         *  We only support 1 site for now until
         *  next release
         */
        siteName = (String)keys.nextElement();
        site = m_spiderManager.getSite(siteName);
        siteStats = new SiteStats(site);
        
        reportData( site, siteStats );
    }
//------------------------------------------------------------------------------    
    private void collectData(){
        
        /*
         *  Starting the process manager
         */
        m_spiderManager.start();
        
        
        /*
         *  We wait until the m_spiderManager is done
         *  before we return from this method
         */
        try{
            m_spiderManager.join();
        }
        catch( InterruptedException intEx ){
            System.err.println( "Metis::collectData() : " + intEx );
        }
        
        /*
         *  Doing a bit of cleanup
         */
        System.gc();
        System.runFinalization();
    }
//------------------------------------------------------------------------------
    private void reportData( Site site, SiteStats stats ){
        MetisReport report = null;
        
        switch( MetisOpt.getReportType() ){
            case MetisGlobal.REPORT_TYPE_ASCII:
                report = (MetisReport)new MetisReportASCII( site, stats, site.getDomain() );
                break;
            case MetisGlobal.REPORT_TYPE_HTML:
                report = (MetisReport)new MetisReportHTML( site, stats );
                break;
            default:
                System.err.println("Unsupported report type. Using general ASCII report\n");
                report = (MetisReport)new MetisReportASCII( site, stats, site.getDomain() );
                break;
        }
        
        report.generateReport();
    }
//------------------------------------------------------------------------------    
//------------------------------------------------------------------------------
    public static void main( String args[] ){
        String site = "";
        Metis metis = null;
        MetisOpt options = new MetisOpt();
        Getopt opt = new Getopt("metis", args, "DRSRr:U:P:T:F:h:");
        int c;
        String argument = "";
        
        if( args.length < 1 )
            options.usage();
        
        while( (c = opt.getopt()) != -1 ){
            argument = "";
            switch( c ){
                case 'D':
                    options.setDebug(true);
                    break;
                case 'R':
                    // not implemented yet
                    break;
                case 'S':
                    // not implemented yet
                    break;
                case 'r':
                    argument = opt.getOptarg();
                    if( argument == null ){
                        System.err.println("You need to specify a request method!");
                        options.usage();
                    }
                    try{
                        options.setRequestMethod( Integer.parseInt(argument) );
                    }
                    catch( NumberFormatException requestNumEx ){
                        System.err.println(requestNumEx);
                        System.exit(1);
                    }
                    break;
                case 'U':
                    argument = opt.getOptarg();
                    if( argument == null ){
                        System.err.println("You need to specify a username!");
                        options.usage();
                    }
                    options.setRequestUserName(argument);
                    break;
                case 'P':
                    argument = opt.getOptarg();
                    if( argument == null ){
                        System.err.println("You need to specify a password!");
                        options.usage();
                    }
                    options.setRequestPassword(argument);
                    break;
                case 'T':
                    argument = opt.getOptarg();
                    if( argument == null ){
                        System.err.println("You need to specify a report type!");
                        options.usage();
                    }
                    try{
                        options.setReportType( Integer.parseInt(argument) );
                    }
                    catch( NumberFormatException reportNumEx ){
                        System.err.println(reportNumEx);
                        System.exit(1);
                    }
                    break;
                case 'h':
                    argument = opt.getOptarg();
                    if( argument == null ){
                        System.err.println("You need to specify a site!");
                        options.usage();
                    }
                    options.setrootURL(argument);
                    break;
                case 'X':
                    argument = opt.getOptarg();
                    if( argument == null ){
                        System.err.println("You need to specify a proxy host!");
                        options.usage();
                    }
                    options.setProxyHost(argument);
                    break;
                case 'C':
                    argument = opt.getOptarg();
                    if( argument == null ){
                        System.err.println("You need to specify a proxy port!");
                        options.usage();
                    }
                    try{
                        Integer proxyPort = new Integer(argument);
                        options.setProxyPort( proxyPort.intValue() );
                    }
                    catch( NumberFormatException numEx ){
                        System.err.println("You need to specify a number for the proxy port ! : " + numEx );
                        options.usage();
                    }
                    break;
                default:
                    System.err.println("Unknown options : " + (char)c);
                    options.usage();
                    break;
            }
        }
        
        //validate the options
        if( !options.validateOptions() ){
            options.usage();
        }
        
        try{
            metis = new Metis();
        }
        catch( MalformedURLException urlEx ){
            System.err.println("Metis::main() : " + urlEx );
            System.exit(0);
        }
        
        metis.run();
        
        System.exit(0);
    }
}
