/*
** Copyright (C) 2001,2002 Sacha Faust <sacha@severus.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*
 *  Version : 1.3
 */
package faust.sacha.web.data;

import faust.sacha.web.util.*;
import java.util.*;
import java.net.MalformedURLException;

public class FileURL extends URLData {    
    private String m_fileExtension;
    private Hashtable m_argumentHash;
    private boolean m_asArguments;
    private int m_argumentIndex;
    //private ArrayList m_queryList;
    private ArrayList m_queryStateList;
    
    public FileURL( URLInfo url ) throws MalformedURLException {
        super(url);

        m_argumentHash = new Hashtable(0);
        m_fileExtension = "";
        m_asArguments = false;
        m_argumentIndex = 0;
        m_queryStateList = new ArrayList(0);
        
        init();
    }
//------------------------------------------------------------------------------
    public FileURL( URLData url ) throws java.net.MalformedURLException{
        this( (URLInfo)url );
        
        m_headers = url.getHeaders();
        // Links
        super.addLinkList( url.getLinks() );
        
        // emails
        super.addEmailList( url.getEmailList() );
        
        // comments
        super.addCommentList( url.getComments() );
        
        // cookies
        super.setCookieInfo( url.getCookieInfo() );
        
        // server
        super.setServerInfo( url.getServerInfo() );
        
        // last modified
        super.setLastModifiedInfo( url.getLastModifiedInfo() );
        
        // content length
        super.setContentLengthInfo( url.getContentLengthInfo() );
        
        // content type
        super.setContentTypeInfo( url.getContentTypeInfo() );
        
        // content location
        super.setContentLocationInfo( url.getContentLocationInfo() );
        
    }
//------------------------------------------------------------------------------
    private void init(){
        setFileExtension( super.getFile() );
        if( getQuery() != "" )
            addQueryData( this );
    }
//------------------------------------------------------------------------------    
    private void setFileExtension( String fileName ){
        int dotIndex;
        
        if( fileName == "" ){
            m_fileExtension = "";
            return;
        }
        
        /*
         *  we try to get the index of the . in the filename part of the url
         *  if no . was found, the file extension is empty
         *  else, we add 1 to the dotIndex so it points to the first letter of the extension
         *  and store the rest of the string to m_fileExtension
         */
        if( (dotIndex = fileName.lastIndexOf(".")) == -1 )
            m_fileExtension = "";
        else{
            dotIndex++;     // setting dotindex to retreive only the extension
            m_fileExtension = fileName.substring(dotIndex);
        }
    }
//------------------------------------------------------------------------------
    private void setArguments( String query ){
        String arguments = query;
        StringTokenizer strToken = null;
        String varAndData = null;
        String var = null;
        String data = null;
        int equalIndex;
        
        if( (arguments == null) || (arguments == "") )
            return;
        else
            m_asArguments = true;
        
        /*
         *  if the argument is only the ?, we return from the method
         *  else, we advance the arguments variable by 1 to get rid of the
         *  ?
         */
        if( arguments == "?" ){
            if( WebGlobal.DEBUG )
                System.err.println("FileURL::setArguments() : single argument ? in : " + getURL() );
            
            return;
        }
        else
            arguments = arguments.substring(1, arguments.length());
        
        // splitting the string to seperate all var=data 
        strToken = new StringTokenizer(arguments, "&");
        
        while( strToken.hasMoreTokens() ){
            varAndData = strToken.nextToken();
            
            if( WebGlobal.DEBUG )
                System.err.println("FileURL::setArguments() : varAndData = " + varAndData);
            
            /*
             *  We find the index of the first =
             *  - if there is none, we know that we only have a variable with no data 
             *    and we store varAndData into var
             *  - else, we store everything before the = in var and everything after = in data
             */
            if( (equalIndex = varAndData.indexOf("=")) == -1 ){
                // no data set to the var
                var = varAndData;
                data = "";
            }
            else{
                var = varAndData.substring(0, equalIndex);
                data = varAndData.substring(equalIndex + 1, varAndData.length());
            }
            
            /*
             *  we always try to add the argument variable.
             *  if it's already there, it wont be added
             */
            addArgumentVar(var);
            addDataToArgumentVar(var, data);            
        }
    }
//------------------------------------------------------------------------------
    public void addArgumentVar( String var ){
        
        if( !m_argumentHash.containsKey(var) )
            m_argumentHash.put( (Object)var, new ArrayList(0) );
    }
//------------------------------------------------------------------------------
    public void addDataToArgumentVar( String var, String data ){
        ArrayList dataList = null;
        
        // if the var is not already a hashtable key, we add it
        if( !m_argumentHash.containsKey( (Object)var ) )
            addArgumentVar( var );
        
        dataList = getArgumentVarDataList(var);
        
        if( dataList == null ){
            System.err.println("FileURL::addDataToArgumentVar() : FATAL ERROR : BAD CODING ERROR ABORTING!. Please report this error");
            System.exit(1);
        }
        
        /*
         *  Logic:
         *  we add the data to the position specified from m_argumentIndex
         *  This will allow to have a global position for argument so full
         *  arguments sessions can be replaid in order after
         *
         *  If the size of the Array is not big enough to old the data at the
         *  right position, we fill it up with empty strings
         */
        while( dataList.size() < m_argumentIndex )
            dataList.add( (Object)"" );
        
        dataList.add( m_argumentIndex , (Object)data );
        
        m_argumentIndex++;
    }
//------------------------------------------------------------------------------
    public ArrayList getArgumentVarDataList( String var ){
       
        if( !m_argumentHash.containsKey((Object)var) )
            return null;
        
        return (ArrayList)m_argumentHash.get( (Object)var );
    }
//------------------------------------------------------------------------------
    public Enumeration getArgumentVarNames(){
        return m_argumentHash.keys();
    }
//------------------------------------------------------------------------------
    public void addQueryData( FileURL scriptData ){
        ScriptStateInfo data = null;
        String query = null;
        
        if( scriptData == null )
            return;
        
        try{
            data = new ScriptStateInfo( (URLData)scriptData );
        }
        catch( MalformedURLException urlEx ){
            System.err.println("FileURL::addQueryData() : " + urlEx.toString());
            return;
        }
        
        query = scriptData.getQuery();
        
        if( (query == null) || (query == "") )
            return;
        
        // add the query so the query info is store
        setArguments(query);
        
        if( isQueryPresent(data.getQuery()) ){
            if( WebGlobal.DEBUG )
                System.err.println("FileURL::addQueryData() : duplicated query key : " + query + " on : " + scriptData.getURL() );
        }
        else{
            m_queryStateList.add( (Object)data );
        }
    }
//------------------------------------------------------------------------------
    public ArrayList getQueryDataList(){
        return m_queryStateList;
    }
//------------------------------------------------------------------------------    
    public boolean isQueryPresent( String queryToCheck ){
        ScriptStateInfo query = null;
        
        if( queryToCheck == null )
            return false;
        
        for( int i = 0; i < m_queryStateList.size(); i++ ){
            query = (ScriptStateInfo)m_queryStateList.get(i);
            if( query.getQuery().compareToIgnoreCase(queryToCheck) == 0 )
                return true;
        }
        return false;
    }
//------------------------------------------------------------------------------    
    public String getExtension(){
        return m_fileExtension;
    }
//------------------------------------------------------------------------------
    public String getFileName(){
        return super.getFile();
    }
//------------------------------------------------------------------------------    
    public boolean isScript(){
        return m_asArguments;
    }
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
    private void DEBUG_DUMP_ARGUMENTS(){
        Enumeration argumentKeys = getArgumentVarNames();
        String varName = null;
        ArrayList dataList = null;
        
        while( argumentKeys.hasMoreElements() ){
            varName = (String)argumentKeys.nextElement();
            System.out.println("Var = " + varName);
            dataList = this.getArgumentVarDataList(varName);
            
            for( int i = 0; i < dataList.size(); i++ )
                System.out.println("\t" + (String)dataList.get(i));
        }        
    }
//------------------------------------------------------------------------------
    /*
    private static void main( String args[] ){
        URLInfo t = new URLInfo("http://ad.doubleclick.net/jump/idg.pixel.global/idgpixel7.18.00;sz=1x1;ord=12092001162221?");
        FileURL s = new FileURL(t);
        
        System.out.println("filename : " + s.getFile() );
        System.out.println("is script before " + s.isScript() );
        System.out.println("query : " + s.getQuery() );
        s.DEBUG_DUMP_ARGUMENTS();
        
    }
     */
//------------------------------------------------------------------------------
}
