/*
** Copyright (C) 2001,2002 Sacha Faust <sacha@severus.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*
 *  Version : 1.3
 */
package faust.sacha.web.bot.spider.data;

import faust.sacha.web.data.URLData;
import faust.sacha.web.data.Site;
import faust.sacha.web.util.WebGlobal;

import faust.sacha.web.bot.spider.data.UrlQ;
import faust.sacha.web.bot.spider.event.ThreadEventManager;
import faust.sacha.web.bot.spider.event.ThreadEvent;

import java.util.*;

public class WaitUrlQ extends ThreadEventManager implements UrlQInt {
    
    private Stack m_urlList;
    private boolean m_gotNotified;
    private Object m_gotNotifiedMutex;
    private Site m_site;
    private UrlQ m_toWaitFor;
    
    public WaitUrlQ( String name, Site site, UrlQ toWait ) {
        super(name);
        m_urlList = new Stack();
        m_gotNotified = true;
        m_site = site;
        m_toWaitFor = toWait;
        m_gotNotifiedMutex = new Object();
    }
//------------------------------------------------------------------------------
    private void notifySiteFinish(){
        ThreadEvent event = null;
        
        /*
         *  Ask the site if we are done with the spider
         */
        event = new ThreadEvent(this, null, ThreadEvent.EVENT_IS_SITE_FINISH);
        sendEvent(event, m_site);
    }
//------------------------------------------------------------------------------
    public void run(){
        super.run();
        
        process();
        
        if( WebGlobal.DEBUG )
            System.err.println( getName() + "::run() : Exiting run()" );
    }
//------------------------------------------------------------------------------
    private synchronized void process(){
    
        int state = 0;
        
        while( true ){
            state = getState();
            
            if( (state == ThreadEvent.EVENT_STOP) || (state == ThreadEvent.EVENT_SITE_FINISH) )
                break;
            else{
                try{
                    /*
                     *  Release the lock and wait to be notified
                     *      - setState() ensure that we will wake up when a new the state is changed     
                     */
                    wait();
                }
                catch( InterruptedException intEx ){
                    if( WebGlobal.DEBUG )
                        System.err.println(getName() + "::run() : " + intEx);
                }
            }
        }
    }
//------------------------------------------------------------------------------
    public synchronized URLData get(){
        URLData retURL = null;
        boolean isEmpty = true;
                
        if( m_urlList.isEmpty() ){
            try{
                /*
                 *  We release the lock and wait for a period of time. This is crucial
                 *  to detect if the spider is done with the site. If the wait()
                 *  timedout, the m_gotNotified will be false. We then wait until
                 *  the m_toWaitFor is empty and we notify the m_site that we
                 *  think that the site is finish. The m_site will send a event
                 *  back to us to confirm, else, we continue and more data will come.
                 */
                wait(1000);
                
                if( !m_gotNotified ){
                    /*
                     *  The wait(int milsec) timedout. This is the indication that
                     *  we might be done spidering the site
                     */
                    
                    /*
                     *  waiting for toWaitFor to be empty or if we our m_urlList is no
                     *  longer empty
                     */
                    while( !m_toWaitFor.isEmpty() ){
                        if( WebGlobal.DEBUG )
                            System.err.println( getName() + "::get() : waiting for m_toWaitFor is empty!" );
                        
                        /*
                         *  We release the lock and wait for half a second
                         *  We then test if our m_urlList is still empty. If so, we continue the loop
                         *  else, we break out of it. and goto pop_element section
                         */                        
                        try{
                            wait(1000);
                        }
                        catch( InterruptedException intEx ){
                            if( WebGlobal.DEBUG )
                                System.err.println( getName() + "::get() : " + intEx );
                        }
                        
                        if( !m_urlList.isEmpty() ){
                            /*
                             *  No matter what people say, goto's are sometimes
                             *  usefull. It's silly that they don't support it under
                             *  Java.
                             */
                            isEmpty = false;
                            break;
                        }
                    }
                    
                    if( isEmpty ){
                        /*
                         *  ask Site if we are done for this site
                         *  The site will reply with an EVENT_SITE_FINISH event if it's
                         *  done.
                         */
                        notifySiteFinish();
                        return null;
                    }
                }
                
                /*
                 *  Resetting the m_gotNotified to false
                 *  to make sure it's eveluated properly
                 *  in the next cycle
                 */
                m_gotNotified = false;
            }
            catch( InterruptedException intEx ){
                if( WebGlobal.DEBUG )
                    System.err.println( getName() + "::get() : " + intEx);
            }
        }
        
        try{
            retURL = (URLData)m_urlList.pop();
            return retURL;
        }
        catch( EmptyStackException emptyEx ){
            return null;
        }
    }
//------------------------------------------------------------------------------
    public synchronized void add(URLData toAdd) {
        
        if( WebGlobal.DEBUG )
            System.err.println( getName() + "::add() : adding : " + toAdd.getURL() );
        
        m_urlList.add(toAdd);
        
        /*
         *  Setting m_gotNotified to true
         *  to let get() know that it got notified and not
         *  that wait() timeout
         */
        m_gotNotified = true;
        
        notifyAll();
    }
//------------------------------------------------------------------------------
    public boolean isEmpty() {
        return m_urlList.isEmpty();
    }
}
