/*
** Copyright (C) 2001,2002 Sacha Faust <sacha@severus.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*
 *  Version : 1.3
 */
package faust.sacha.web.bot.spider;

import faust.sacha.web.data.*;
import faust.sacha.web.util.*;
import faust.sacha.web.bot.spider.event.*;
import faust.sacha.web.bot.spider.data.*;
import faust.sacha.web.bot.spider.util.*;
import java.util.*;
import java.net.MalformedURLException;

public class SpiderManager extends ThreadEventManager {

    private Hashtable m_siteList;
    private int m_nbSiteRunning;
    private Object m_nbSiteRunningMutex;
    private String m_initialSite;
    /*
     *  Used to know if we print a message when we are done with a site
     *  or not.
     *      - Metis use this to tell the user that one site is done in the console
     */
    public static boolean PRINT_SITE_START_MESSAGE = false;
    public static boolean PRINT_SITE_FINISH_MESSAGE = false;
    
    
    public SpiderManager( String initialSite ) {
        super("SpiderManager");
        m_siteList = new Hashtable();
        m_nbSiteRunning = 0;
        m_nbSiteRunningMutex = new Object();
        m_initialSite = initialSite;
    }
//------------------------------------------------------------------------------
    public SpiderManager( URLInfo initialSite ){
        super( initialSite.getBaseURL() );
    }
//------------------------------------------------------------------------------
    private synchronized void DEBUG_dumSpiderInfo(){
        Enumeration keys = null;
        String keyName = null;
        Site site = null;
        ArrayList urlsDone = null;
        URLInfo u = null;
        
        keys = m_siteList.keys();
        
        while( keys.hasMoreElements() ){
            keyName = (String)keys.nextElement();
            site = (Site)m_siteList.get(keyName);
            urlsDone = (ArrayList)site.getUrlsDone();
            
            System.err.println("Site : " + site.getDomain() );
            for( int i = 0; i < urlsDone.size(); i++ ){
                u = (URLInfo)urlsDone.get(i);
                System.err.println("\t" + i + " : " + u.getURL());
            }
            
            System.err.println();
        }
    }
//------------------------------------------------------------------------------
    public void run(){
        super.run();
       
        /*
         *  Load the initial site
         */
        addSiteAndSpider(m_initialSite);

        process();
        
        
        // 
        //DEBUG_dumSpiderInfo();
       
    }
//------------------------------------------------------------------------------
    private synchronized void process(){
        int state = 0;
        ThreadEvent eventStop = null;
        
        while( true ){
            state = getState();
            
            if( (state == ThreadEvent.EVENT_STOP) || (getNbRunningSites() == 0) )
                break;
            else{
                /*
                 *  Release the lock and wait to be notified
                 *      - setState() ensure that we will be notified on state change
                 */
                try{
                    wait();
                }
                catch( InterruptedException intEx ){
                    if( WebGlobal.DEBUG )
                        System.err.println( getName() + "::process : " + intEx );
                }
            }
        }
        
        if( getState() == ThreadEvent.EVENT_STOP ){
            stopAllListenningThreads();
        }
    }
//------------------------------------------------------------------------------
    private void addSiteAndSpider( String site ){
        Site toAdd = null;
        
        try{
            toAdd = new Site( new URLInfo(site), this );
        }
        catch( MalformedURLException urlEx ){
            System.err.println( getName() + "::addSite : " + urlEx );
            return;
        }  
        
        /*
         *  Adding the site to our site list
         */
        addSite(toAdd);
        
        /*
         *  Add the new site to our listenners.
         *  this enable us to send event like EVENT_STOP
         */
        addToListeners( (Object)toAdd );
        
        /*
         *  Add ourselve to the new Site listenners
         *  so it can tell us when he's done
         */
        toAdd.addToListeners( (Object)this );
        
        /*
         *  adding 1 to our number of current site
         */
        addRunningSite();
        
        /*
         *  Starting the site thread and the spider
         */
        
        if( PRINT_SITE_START_MESSAGE )
            System.out.println( "Starting spider on : " + toAdd.getDomain() );
            
        toAdd.start();
    }
//------------------------------------------------------------------------------
    private void addRunningSite(){
        synchronized( m_nbSiteRunningMutex ){
            m_nbSiteRunning++;
        }
        
        if( WebGlobal.DEBUG )
            System.err.println( getName() + "::addRunningSite : m_nbSiteRunning : " + m_nbSiteRunning );
        
    }
//------------------------------------------------------------------------------
    private synchronized void removeRunningSite(){
        if( m_nbSiteRunning <= 0 ){
            System.err.println( "CODE ERROR->" + getName() + "::removeRunningSite : m_nbSiteRunning lower then 0! : " + m_nbSiteRunning );
            
            return;
        }
            
            
        if( m_nbSiteRunning > 0 )
            m_nbSiteRunning--;
        
        if( WebGlobal.DEBUG )
            System.err.println( getName() + "::removeRunningSite : m_nbSiteRunning : " + m_nbSiteRunning );
        
        /*
         *  Notify all that a site as just finish
         *      - process() needs this
         */
        notifyAll();
    }
//------------------------------------------------------------------------------
    public int getNbRunningSites(){
        return m_nbSiteRunning;
    }
//------------------------------------------------------------------------------
    private void addSite( Site toAdd ){
        
        synchronized( m_siteList ){
            m_siteList.put( (Object)toAdd.getDomain(), (Object)toAdd );
        }
        
        if( WebGlobal.DEBUG )
            System.err.println( getName() + "::addSite : added site : " + toAdd.getDomain() ); 
    }
//------------------------------------------------------------------------------
    public void receiveEvent( ThreadEvent event ){
        Object result = null;
        Object source = null;
        String sourceName = null;
        int eventType = 0;
        
        result = event.getResult();
        source = event.getSource();
        sourceName = ((ThreadEventManager)source).getName();        
        eventType = event.getEventType();
        
        switch( eventType ){
            case ThreadEvent.EVENT_SPIDER_SITE:
                if( WebGlobal.DEBUG )
                    System.err.println(getName() + "::receiveEvent : EVENT_SPIDER_SITE  from : " + source.getClass().getName() );
                
                if( result == null ){
                    System.err.println("CODE ERROR->" + getName() + "::receiveEvent : got null result on event : EVENT_SPIDER_SITE from : " + source.getClass().getName() );
                    
                    break;
                }
                else if( !(result instanceof String) ){
                    System.err.println("CODE ERROR->" + "::receiveEvent : event type : EVENT_SPIDER_SITE: data type of: " + result.getClass().getName() );
                    
                    break;
                }
                else{
                    addSiteAndSpider( (String)result );
                    break;
                }
            case ThreadEvent.EVENT_FINISH:
                /*
                 *  This is one of our FileGetter thread reporting that he's done
                 */
                if( WebGlobal.DEBUG )
                    System.err.println( getName() + "::receiveEvent : receive EVEN_FINISH FROM : " + ((ThreadEventManager)source).getName() );
                
                if( result != null ){
                    System.err.println("CODE_ERROR->" + getName() + "::receiveEvent : event type : EVENT_FINISHH : data is not NULL and is type of: " + result.getClass().getName() + " from : " + sourceName );
                    break;
                }
                else if( !(source instanceof Site) ){
                    System.err.println( "CODE ERROR->" + getName() + "::receiveEvent : event type : EVENT_TO_SET_INFO: data type of: " + result.getClass().getName() + " from : " + source.getClass().getName() + " : " + sourceName );
                    break;
                }
                else{
                    if( PRINT_SITE_FINISH_MESSAGE )
                        System.out.println( "Done with site : " + ((Site)source).getDomain() );
                    
                    removeRunningSite();
                    break;
                }                
            default:
                //pass the message to the parent                
                super.receiveEvent(event);
                break;
        }
    }
//------------------------------------------------------------------------------
    public synchronized Enumeration getSiteNames(){
        return m_siteList.keys();
    }
//------------------------------------------------------------------------------
    public synchronized Site getSite( String siteName ){
        return ( (Site)m_siteList.get( (Object)siteName ) );
    }
//------------------------------------------------------------------------------
}
