/*
*
* $Author: sacha Faust $
* $Revision: 1.1 $
* $Log: rootdse.c,v $
* Revision 1.1  2001-08-25 20:18:25-04  sacha
* Initial revision
*
*/

#include <stdio.h>
#include <stdlib.h>
#include "ldap.h"

#ifdef _GNUC_
#include <unistd.h>
#else
#include "getopt.h"
#endif

#define BANNER		"LDAPRootDSE v.1.0 by Sacha Faust (sacha@smugline.net)"
#define MIN_PORT	1
#define MAX_PORT	6535

#define SUCCESS 	1
#define ERROR		0


struct ldap_connection_info{
	char *host;
	int port;
	char *binddn;
	char *password;
};

int get_rootdse( struct ldap_connection_info * );
LDAP *get_ldap_for_rootdse( struct ldap_connection_info * );
void usage( char * );

int main( int argc, char *argv[] ){

	struct ldap_connection_info connection;
	int o;

	if( argc < 2 )
		usage(argv[0]);

	memset(&connection, 0, sizeof(struct ldap_connection_info));

	while( (o = getopt(argc, argv,"h:b:p:P:")) != EOF ){
		switch(o){
			case 'h':
				connection.host = optarg;
				break;

			case 'b':
				connection.binddn = optarg;
				break;

			case 'p':
				connection.password = optarg;
				break;

			case 'P':
				connection.port = atoi(optarg);
				if( (connection.port < MIN_PORT) || (connection.port > MAX_PORT) ){
					fprintf(stderr, "Invalid port specified : %d\n", connection.port );
					exit(1);
				}
				break;

			default:
				fprintf(stderr, "Unknown option specified : %s\n", optarg);
				usage(argv[0]);
				break;
		}
	}

	if( connection.host == NULL ){
		fprintf(stderr, "You need to specify a host\n" );
		usage(argv[0]);
	}

	if( get_rootdse(&connection) == ERROR ){
		fprintf( stderr, "Unable to get rootdse for : %s\n", connection.host );
		exit(1);
	}

	return 0;
}

int get_rootdse( struct ldap_connection_info *connection ){

	LDAP *ld = NULL;
	LDAPMessage *ldmsg = NULL;
	BerElement *ber = NULL;
	char *szattrib;
	char **szvalues;
	int msgid;
	int i;


	if( (ld = get_ldap_for_rootdse(connection)) == NULL ){
		fprintf(stderr, "Error setting ld for rootdse\n");
		return ERROR;
	}

	if( ldap_simple_bind_s(ld, connection->binddn, connection->password) != LDAP_SUCCESS ){
		ldap_perror(ld, "ldap_simple_bind_s");
		ldap_unbind(ld);
		return ERROR;
	}

	if( (msgid = ldap_search(ld, "", LDAP_SCOPE_BASE, "(objectclass=*)", NULL, 0)) == -1 ){
		ldap_perror(ld, "ldap_search");
		ldap_unbind(ld);
		return ERROR;
	}

	while( ldap_result(ld, msgid, 0, NULL, &ldmsg) == LDAP_RES_SEARCH_ENTRY ){
		for( szattrib = ldap_first_attribute(ld, ldmsg, &ber); szattrib != NULL; szattrib = ldap_next_attribute(ld, ldmsg, ber) ){
			szvalues = ldap_get_values(ld, ldmsg, szattrib);
			printf("%s\n", szattrib);
			for( i = 0; szvalues[i] != NULL; i++ )
				printf("\t%s\n", szvalues[i]);

			ldap_value_free(szvalues);
			ldap_memfree(szattrib);
		}
		ber_free(ber, 0);
		ldap_msgfree(ldmsg);
	}

	if( ldap_get_lderrno(ld, NULL, NULL) != LDAP_SUCCESS ){
		ldap_perror(ld, "ldap_get_lderrno");
		ldap_unbind(ld);
		return ERROR;
	}

	return SUCCESS;
}

LDAP *get_ldap_for_rootdse( struct ldap_connection_info *connection ){

	LDAP *ld;
	int proto_version = LDAP_VERSION3;


	if( (ld = ldap_init(connection->host, connection->port)) == NULL ){
		ldap_perror(ld, "ldap_init\n");
		return NULL;
	}

	if( ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &proto_version) != LDAP_SUCCESS ){
		ldap_perror(ld, "ldap_set_option : protocol version\n");
		ldap_unbind(ld);
		return NULL;
	}

	if( ldap_set_option(ld, LDAP_OPT_REFERRALS, LDAP_OPT_OFF) != LDAP_SUCCESS ){
		ldap_perror(ld, "ldap_set_option : referrals\n");
		ldap_unbind(ld);
		return NULL;
	}

	return ld;
}

void usage( char *progname ){
	fprintf(stderr, "%s\n", BANNER );
	fprintf(stderr, "Usage : %s [-h host] options\n", progname );
	fprintf(stderr, "Options :\n");
	fprintf(stderr, "\t[-b binddn]\n");
	fprintf(stderr, "\t[-p password]\n");
	fprintf(stderr, "\t[-P port]\n");

	exit(1);
}
